/*  Aktualisiert die Ausschaltzeiten (Rahmenarbeitszeiten, arbeitsfreie Wochentage) der gegebenen Ressource für den
    vorgegebenen Zeitraum.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__renew_offtime', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__renew_offtime(
      _resource_id      integer,
      _timeframe_start  timestamp DEFAULT scheduling.resource_timeline__offtime__default_timeframe_start__get(),
      _timeframe_end    timestamp DEFAULT scheduling.resource_timeline__offtime__default_timeframe_end__get(),
      _loglevel         int       DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS void AS $function$
  DECLARE

      _resource_record scheduling.resource;
      _working_days    integer[7];
      _shift_times     time[2];

    BEGIN

      -- handling only for ksvba resources implemented
      SELECT r.*
        INTO _resource_record
        FROM scheduling.resource AS r
       WHERE r.id = _resource_id;

      -- validate resource_record
      IF ( _resource_record IS NULL ) THEN
          RAISE EXCEPTION '%', format( 'resource id %L not found', _resource_id );
      END IF;

      IF ( _resource_record.context <> 'ksvba' ) THEN
          RAISE EXCEPTION 'renew offtime only for ksvba implemented';
      END IF;

      IF ( _timeframe_end <= _timeframe_start ) THEN
          RAISE NOTICE 'invalid timeranges for offtime renewal';
      END IF;

      _working_days := scheduling.ksvba__get_workingdays( _ksvba_id => _resource_record.context_id );
      _shift_times  := scheduling.ksvba__get_shifttimes( _ksvba_id => _resource_record.context_id );

      -- generate blocked times into temp table for comparison
      CREATE TEMP TABLE _blockedTimes AS
          SELECT *
               , _resource_id
            FROM scheduling.resource_timeline__blocked_timeslots__generate(
                    _shift_start_time   => _shift_times[1],
                    _shift_end_time     => _shift_times[2],
                    _working_days       => _working_days,
                    _startDate          => _timeframe_start,
                    _stopDate           => _timeframe_end,
                    _ta_fk              => _resource_record.ta_fk
                 )
      ;

      -- remove all contained offtimes, since they will be replaced
      DELETE FROM scheduling.resource_timeline
       WHERE
             -- this is explicitly the contains operator to delete the inner blocked dates, not the edges,
             -- since we need to update those
                tsrange( ( SELECT min( _blockedTimes.date_start ) FROM _blockedTimes ), ( SELECT max( _blockedTimes.date_end ) FROM _blockedTimes ), '[]' )
             @> tsrange( ti_date_start                                                , ti_date_end                                                , '[]' )
         AND ti_resource_id = _resource_id
         AND ti_type IN ( 'off.day', 'off.time' )
      ;

      -- update lower edge
      UPDATE scheduling.resource_timeline
         SET ti_date_end = date_trunc( 'day', ( SELECT min( _blockedTimes.date_start ) FROM _blockedTimes ) )
       WHERE
             -- select resource offtimes which overlaps with the new data and needs to be adjuested
             ti_resource_id = _resource_id
         AND ti_type IN ( 'off.day', 'off.time' )
         AND tsrange( ( SELECT min( _blockedTimes.date_start ) FROM _blockedTimes ), ( SELECT max( _blockedTimes.date_end ) FROM _blockedTimes ), '[]' )
          && tsrange( ti_date_start                                                , ti_date_end                                                , '[]' )
         AND ti_date_start < ( SELECT min( _blockedTimes.date_start ) FROM _blockedTimes )
      ;

      -- update upper edge
      UPDATE scheduling.resource_timeline
         SET ti_date_start = date_trunc( 'day', ( SELECT max( _blockedTimes.date_end ) FROM _blockedTimes ) )
       WHERE
             ti_resource_id = _resource_id
         AND ti_type IN ( 'off.day', 'off.time' )
         AND tsrange( ( SELECT min( _blockedTimes.date_start ) FROM _blockedTimes ), ( SELECT max( _blockedTimes.date_end ) FROM _blockedTimes ), '[]' )
          && tsrange( ti_date_start                                                , ti_date_end                                                , '[]' )
         AND ti_date_end > ( SELECT max( _blockedTimes.date_end ) FROM _blockedTimes )
      ;

      -- insert middle part
      INSERT INTO scheduling.resource_timeline
                  ( ti_date_start, ti_date_end, ti_type, ti_resource_id )
           SELECT *
             FROM _blockedTimes;

      DROP TABLE _blockedTimes;

      -- Arbeitsfreie Tage mit zugewiesener Kapazität in Arbeitstage umwandeln.
      PERFORM scheduling.resource_timeline__rewrite_offtime(
                  _resource_id  => _resource_id,
                  _date         => ti_date_start::date,
                  _toworkday    => true,
                  _loglevel     => _loglevel
              ) -- Umwandlung in Arbeitstag.
         FROM scheduling.resource_timeline ti
        CROSS JOIN LATERAL tplanterm.ks_day_kapa( _resource_id => ti.ti_resource_id, _date => ti.ti_date_start::date, EXCEPTION_IF_NO_KAPA => false ) AS day_kapa
        WHERE ti_type = 'off.day'                                              -- Alle Off.Day-Einträge in der Timeline der Ressource im vorgebenen Zeitfenster durchgehen, ...
          AND day_kapa > 0                                                     -- ... welche an einem Arbeitstag liegen.
          AND ti_resource_id = _resource_id
          AND ti_date_start < _timeframe_end
          AND ti_date_end   > _timeframe_start;

      -- Arbeitstage mit gelöschter Kapazität in arbeitsfreie Tage umwandeln.
      PERFORM scheduling.resource_timeline__rewrite_offtime(
                  _resource_id  => _resource_id,
                  _date         => ti_date_start::date,
                  _toworkday    => false,
                  _loglevel     => _loglevel
              ) -- Umwandlung in Arbeitstag.
         FROM scheduling.resource_timeline ti
        CROSS JOIN LATERAL tplanterm.ks_day_kapa( _resource_id => ti.ti_resource_id, _date => ti.ti_date_start::date, EXCEPTION_IF_NO_KAPA => false ) AS day_kapa
        WHERE ti_type = 'off.time'                                              -- Alle Off.Time-Einträge in der Timeline der Ressource im vorgebenen Zeitfenster durchgehen, ...-
          AND day_kapa = 0                                                      -- ... welche an einem arbeitsfreiem Tag liegen.
          AND ti_resource_id = _resource_id
          AND ti_date_start < _timeframe_end
          AND ti_date_end   > _timeframe_start;

      -- Lösche Ausschaltzeiten, welche die Länge 0 haben (Start = Ende).
      DELETE FROM scheduling.resource_timeline
       WHERE ti_resource_id = _resource_id
         AND ti_type IN ( 'off.day', 'off.time' )
         AND ti_date_start < _timeframe_end
         AND ti_date_end   > _timeframe_start
         AND ti_date_start = ti_date_end -- Start = Ende, Länge 0
      ;

  END $function$ LANGUAGE plpgsql;


/*  Aktualisiert die Ausschaltzeiten (Rahmenarbeitszeiten, arbeitsfreie Wochentage) für den gegebenen Filter der
    KSVBA-Shorthands für den vorgegebenen Zeitraum.

    Damit lassen sich mit einem Funktionsaufruf die Ausschaltzeiten für mehrere KSVBA-Ressourcen aktualisieren. Dies erfolgt
    nacheinander mittels einer for-Schleife.
*/
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__renew_offtime(
      _ksb_shorthands   varchar,
      _timeframe_start  timestamp DEFAULT now(),
      _timeframe_end    timestamp DEFAULT scheduling.resource_timeline__offtime__default_timeframe_end__get(),
      _loglevel         int       DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS void AS $function$
  DECLARE

      _resource scheduling.resource;
      _count integer;
      _totalcount integer;

      -- Messung Dauer für die Schleifendurchläufe
      _loop_time_begin    timestamp;
      _loop_time_end      timestamp;
      _loop_time_current  interval;
      _loop_time_min      interval;
      _loop_time_max      interval;
      _loop_time_avg      interval;
      _loop_time_total    interval;

      -- Exception-Handling
      _errmsg text;
      _message_text text;
      _pg_exception_context text;
      _pg_exception_detail text;
      _pg_exception_hint text;

  BEGIN

      _count := 0;
      _totalcount := count( * )
                          FROM scheduling.resource AS r
                          JOIN ksvba AS k ON k.ksb_id = r.context_id
                                          AND r.context = 'ksvba'
                          WHERE   upper( k.ksb_ks_shorthand ) LIKE upper( coalesce( _ksb_shorthands, '' ) )
                      ;

      FOR _resource IN
          SELECT r.*
          FROM scheduling.resource AS r
          JOIN ksvba AS k ON k.ksb_id = r.context_id
                        AND r.context = 'ksvba'
          WHERE   upper( k.ksb_ks_shorthand ) LIKE upper( coalesce( _ksb_shorthands, '' ) )
          ORDER BY r.id
      LOOP

          BEGIN

              _count := _count + 1;
              _loop_time_begin := clock_timestamp();

              IF _loglevel >= 4 THEN
                  RAISE NOTICE '[% / %] start resource_id: %;', _count, _totalcount, _resource.id;
              END IF;

              -- Ausschaltzeiten der Ressource neu anlegen
              PERFORM scheduling.resource_timeline__renew_offtime(
                          _resource_id        => _resource.id,
                          _timeframe_start    => _timeframe_start,
                          _timeframe_end      => _timeframe_end,
                          _loglevel           => _loglevel
                      )
              ;

          EXCEPTION
            WHEN OTHERS THEN
                GET STACKED DIAGNOSTICS _message_text = MESSAGE_TEXT
                                      , _pg_exception_detail = PG_EXCEPTION_DETAIL
                                      , _pg_exception_hint = PG_EXCEPTION_HINT
                                      , _pg_exception_context = PG_EXCEPTION_CONTEXT;
                _errmsg :=  'ERROR: ' || _message_text || E'\n' ||
                            'Detail:' || _pg_exception_detail || E'\n' ||
                            'Context:' || _pg_exception_context || E'\n' ||
                            'Hint:' || _pg_exception_hint || E'\n';
                RAISE WARNING '%', _errmsg;
          END;

          -- Messung Dauer für die Schleifendurchläufe
          _loop_time_end := clock_timestamp();
          _loop_time_current := ( _loop_time_end - _loop_time_begin );
          -- Min
          IF _loop_time_min IS null OR _loop_time_min > _loop_time_current  THEN
              _loop_time_min := _loop_time_current;
          END IF;
          -- Max
          IF _loop_time_max IS null OR _loop_time_max < _loop_time_current THEN
              _loop_time_max := _loop_time_current;
          END IF;
          -- Total
          IF _loop_time_total IS null THEN
              _loop_time_total := _loop_time_current;
          ELSE
              _loop_time_total := _loop_time_total + _loop_time_current;
          END IF;
          IF _loglevel >= 5 THEN
              -- Debug: Duration
              RAISE NOTICE 'duration of [resource_id: %] : %', _resource.id, _loop_time_current;
              -- Debug: Duration Statistics
              RAISE NOTICE 'duration statistics : min:%, avg:%, max:%;', _loop_time_min, ( _loop_time_total / _count ), _loop_time_max;
          END IF;

      END LOOP;

  END $function$ LANGUAGE plpgsql;